#!/usr/bin/env bash

set -euo pipefail  # Enable strict error handling

usage() {
    echo "Usage: $0 [env_file]"
    echo
    echo "env_file: Path to the main environment file. Defaults to 'published.env'."
    exit 1
}

error_log() {
    echo "[ERROR] $1" >&2  # Log errors to stderr
}

DEBUG=${DEBUG:-0}
debug_log() {
    if [[ "$DEBUG" == "true" ]]; then
        echo "[DEBUG] $1" >&2  # Output debug messages if DEBUG is set
    fi
}

try() {
    local exit_code=0
    "$@" || exit_code=$?
    return $exit_code
}

load_env_file() {
    local env_file="$1"

    if [[ ! -f $env_file ]]; then
        debug_log "File $env_file not found. Skipping."
        return
    fi

    debug_log "Loading environment variables from $env_file"

    while read -r line || [[ -n "$line" ]]; do
        line="${line#"${line%%[![:space:]]*}"}"  # Trim leading whitespace

        if [[ -z "$line" || "$line" =~ ^# ]]; then
            debug_log "Skipping line: $line"
            continue
        fi

        debug_log "Processing line: $line"

        if [[ "$line" =~ ^([^=]+)=(.*)$ ]]; then
            name="${BASH_REMATCH[1]}"
            value="${BASH_REMATCH[2]}"
        else
            error_log "Invalid line format: $line"
            continue
        fi

        debug_log "Setting $name to $value"
        export "$name=$value"
    done < "$env_file"
}

# Main script
env_file=${1:-published.env}
user_env_file=${2:-published.user.env}

load_env_file "$env_file"
load_env_file "$user_env_file"

debug_log "Environment variables set successfully."

# Create data directories, it is required for sql container to work in Rancher Desktop
mkdir -p "$GX_VOLUMES_BASE_PATH/data/mssql/data"
mkdir -p "$GX_VOLUMES_BASE_PATH/data/kbs"

debug_log "Starting docker-compose"
docker compose up gxweb --wait
